<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Car;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class CarsController extends Controller
{
    public function index()
    {
        $cars = Car::latest()->paginate(10);
        return view('admin.cars.index', compact('cars'));
    }

    public function create()
    {
        return view('admin.cars.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'make' => 'required|string',
            'model' => 'required|string',
            'vin' => 'nullable|string',
            'stock_number' => 'nullable|string',
            'year' => 'required|integer',
            'price' => 'required|numeric',
            'market_avg_price' => 'nullable|numeric',
            'type' => 'required|string',
            'description' => 'nullable|string',
            'image' => 'nullable|image|max:2048',
            'image_alt' => 'required_with:image|nullable|string',
            'fuel_type' => 'nullable|string',
            'transmission' => 'nullable|string',
            'mileage' => 'nullable|string',
            'ext_color' => 'nullable|string',
            'int_color' => 'nullable|string',
            'engine' => 'nullable|string',
            'drive_type' => 'nullable|string',
            'is_featured' => 'boolean',
            'status' => 'required|string',
            'seller_name' => 'nullable|string',
            'slug' => 'nullable|string|unique:cars,slug',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string',
        ]);

        if ($request->hasFile('image')) {
            $path = \App\Services\ImageService::uploadAndOptimize($request->file('image'), 'cars');
            $validated['image'] = asset('storage/' . $path);
        }

        $validated['is_featured'] = $request->has('is_featured');
        Car::create($validated);

        return redirect()->route('admin.cars.index')->with('success', 'Car added successfully.');
    }

    public function edit($id)
    {
        $car = Car::findOrFail($id);
        return view('admin.cars.edit', compact('car'));
    }

    public function update(Request $request, $id)
    {
        $car = Car::findOrFail($id);
        
        $validated = $request->validate([
            'make' => 'required|string',
            'model' => 'required|string',
            'vin' => 'nullable|string',
            'stock_number' => 'nullable|string',
            'year' => 'required|integer',
            'price' => 'required|numeric',
            'market_avg_price' => 'nullable|numeric',
            'type' => 'required|string',
            'description' => 'nullable|string',
            'image' => 'nullable|image|max:2048',
            'image_alt' => 'required_with:image|nullable|string',
            'fuel_type' => 'nullable|string',
            'transmission' => 'nullable|string',
            'mileage' => 'nullable|string',
            'ext_color' => 'nullable|string',
            'int_color' => 'nullable|string',
            'engine' => 'nullable|string',
            'drive_type' => 'nullable|string',
            'is_featured' => 'sometimes|boolean',
            'status' => 'required|string',
            'seller_name' => 'nullable|string',
            'slug' => 'nullable|string',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string',
        ]);

        if ($request->hasFile('image')) {
            $path = \App\Services\ImageService::uploadAndOptimize($request->file('image'), 'cars');
            $validated['image'] = asset('storage/' . $path);
        }

        $validated['is_featured'] = $request->has('is_featured');
        $car->update($validated);

        return redirect()->route('admin.cars.index')->with('success', 'Car updated successfully.');
    }

    public function destroy($id)
    {
        $car = Car::findOrFail($id);
        $car->delete();
        return redirect()->route('admin.cars.index')->with('success', 'Car deleted successfully.');
    }
}
