<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\FinancingApplication;
use App\Models\ContactMessage;
use App\Models\ServiceBooking;
use App\Models\NewsletterSubscriber;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class FormSubmissionController extends Controller
{
    public function submitFinancing(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'first_name' => 'required|string|max:255',
            'last_name' => 'required|string|max:255',
            'email' => 'required|email',
            'daytime_phone' => 'required|string',
            'address' => 'required|string',
            'city' => 'required|string',
            'state' => 'required|string',
            'zip_code' => 'required|string',
            'country' => 'required|string',
            'consent_given' => 'required|accepted',
        ]);

        if ($validator->fails()) {
            return response()->json(['success' => false, 'errors' => $validator->errors()], 422);
        }

        $data = $request->except(['ssn', '_token']);
        // Store only last 4 digits of SSN for security
        if ($request->filled('ssn')) {
            $ssn = preg_replace('/[^0-9]/', '', $request->ssn);
            $data['ssn_last4'] = substr($ssn, -4);
        }
        $data['ip_address'] = $request->ip();
        $data['application_type'] = $request->input('application_type', 'car_sale');

        $application = FinancingApplication::create($data);

        return response()->json([
            'success' => true,
            'message' => 'Your financing application has been submitted successfully! We will contact you within 24 hours.',
            'application_id' => $application->id,
        ]);
    }

    public function submitContact(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|email',
            'message' => 'required|string|min:10',
        ]);

        if ($validator->fails()) {
            return response()->json(['success' => false, 'errors' => $validator->errors()], 422);
        }

        ContactMessage::create([
            'name' => $request->name,
            'email' => $request->email,
            'phone' => $request->phone,
            'subject' => $request->subject,
            'message' => $request->message,
            'ip_address' => $request->ip(),
        ]);

        return response()->json(['success' => true, 'message' => 'Your message has been sent! We will get back to you soon.']);
    }

    public function submitBooking(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'customer_name' => 'required|string|max:255',
            'customer_email' => 'required|email',
            'customer_phone' => 'required|string',
            'service_type' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json(['success' => false, 'errors' => $validator->errors()], 422);
        }

        ServiceBooking::create($request->only([
            'customer_name', 'customer_email', 'customer_phone',
            'vehicle_make', 'vehicle_model', 'vehicle_year',
            'service_type', 'preferred_date', 'preferred_time', 'notes'
        ]));

        return response()->json(['success' => true, 'message' => 'Booking confirmed! We will contact you to confirm your appointment.']);
    }

    public function subscribeNewsletter(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|email',
        ]);

        if ($validator->fails()) {
            return response()->json(['success' => false, 'errors' => $validator->errors()], 422);
        }

        NewsletterSubscriber::updateOrCreate(
            ['email' => $request->email],
            ['name' => $request->name, 'is_active' => true]
        );

        return response()->json(['success' => true, 'message' => 'Successfully subscribed to our newsletter!']);
    }
}
