<?php

namespace App\Http\Controllers;

use App\Models\Page;
use App\Models\Service;
use App\Models\Car;
use App\Models\Blog;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Response;

class FrontendController extends Controller
{
    private function getSeoData($slug)
    {
        $page = Page::where('slug', $slug)->first();
        if (!$page) return [];

        return [
            'meta_title' => $page->meta_title ?: $page->title,
            'meta_description' => $page->meta_description,
            'canonical_url' => $page->canonical_url,
            'og_title' => $page->og_title,
            'og_description' => $page->og_description,
            'og_image' => $page->og_image,
            'schema_json' => $page->schema_json,
            'is_index' => $page->is_index,
        ];
    }

    public function index()
    {
        $seo = $this->getSeoData('home');
        // Fetch featured cars OR the latest 6 available cars if featured are missing
        $featuredCars = Car::where('is_featured', true)->where('status', 'available')->take(3)->get();
        if ($featuredCars->count() < 3) {
            $latestCars = Car::where('status', 'available')->latest()->take(6)->get();
        } else {
            $latestCars = Car::where('status', 'available')->whereNotIn('id', $featuredCars->pluck('id'))->latest()->take(3)->get();
        }
        
        $featuredServices = Service::where('is_active', true)->take(3)->get();
        
        $contents = \App\Models\PageContent::where('page_name', 'index.php')->get()->pluck('content_value', 'section_key');
        return view('home', array_merge($seo, compact('featuredCars', 'latestCars', 'contents', 'featuredServices')));
    }

    public function about()
    {
        $seo = $this->getSeoData('about');
        return view('about', $seo);
    }

    public function contact()
    {
        $seo = $this->getSeoData('contact');
        return view('contact', $seo);
    }

    public function services()
    {
        $seo = $this->getSeoData('services');
        $services = Service::where('is_active', true)->get();
        return view('services.index', array_merge($seo, compact('services')));
    }

    public function serviceDetail($slug)
    {
        $service = Service::where('slug', $slug)->firstOrFail();
        
        $seo = [
            'meta_title' => $service->meta_title ?: $service->title . ' - Services',
            'meta_description' => $service->meta_description,
            'canonical_url' => $service->canonical_url,
            'og_title' => $service->og_title,
            'og_description' => $service->og_description,
            'og_image' => $service->og_image,
            'schema_json' => $service->schema_json,
            'is_index' => $service->is_index,
        ];

        return view('services.show', array_merge($seo, compact('service')));
    }

    public function cars()
    {
        $seo = $this->getSeoData('cars');
        $cars = Car::where('status', 'available')->paginate(9);
        return view('cars.index', array_merge($seo, compact('cars')));
    }

    public function carDetail($slug)
    {
        $car = Car::where('slug', $slug)->firstOrFail();

        $seo = [
            'meta_title' => $car->meta_title,
            'meta_description' => $car->meta_description,
            'canonical_url' => $car->canonical_url,
            'og_title' => $car->og_title,
            'og_description' => $car->og_description,
            'og_image' => $car->og_image,
            'schema_json' => $car->schema_json,
            'is_index' => $car->is_index,
        ];

        return view('cars.show', array_merge($seo, compact('car')));
    }

    public function sitemap()
    {
        $services = Service::where('is_active', true)->get();
        $blogs = Blog::where('status', 'published')->get();
        $cars = Car::where('status', 'available')->get();
        $pages = Page::all();

        $xml = view('sitemap', compact('services', 'blogs', 'cars', 'pages'))->render();

        return Response::make($xml, 200, ['Content-Type' => 'text/xml']);
    }

    public function sitemapIndex()
    {
        $xml = '<?xml version="1.0" encoding="UTF-8"?>
<sitemapindex xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">
    <sitemap>
        <loc>' . url('/sitemap.xml') . '</loc>
        <lastmod>' . now()->toAtomString() . '</lastmod>
    </sitemap>
</sitemapindex>';

        return Response::make($xml, 200, ['Content-Type' => 'text/xml']);
    }
}
